<?php
/* Copyright (C) 2020 SuperAdmin <francis.appels@z-application.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

/**
 * \file    blockoutstanding/class/actions_blockoutstanding.class.php
 * \ingroup blockoutstanding
 * \brief   Example hook overload.
 *
 * Put detailed description here.
 */

/**
 * Class ActionsBlockOutstanding
 */
class ActionsBlockOutstanding
{
	/**
	 * @var DoliDB Database handler.
	 */
	public $db;

	/**
	 * @var string Error code (or message)
	 */
	public $error = '';

	/**
	 * @var array Errors
	 */
	public $errors = array();


	/**
	 * @var array Hook results. Propagated to $hookmanager->resArray for later reuse
	 */
	public $results = array();

	/**
	 * @var string String displayed by executeHook() immediately after return
	 */
	public $resprints;


	/**
	 * Constructor
	 *
	 *  @param		DoliDB		$db      Database handler
	 */
	public function __construct($db)
	{
		$this->db = $db;
	}


	/**
	 * Execute action
	 *
	 * @param	array			$parameters		Array of parameters
	 * @param	CommonObject    $object         The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
	 * @param	string			$action      	'add', 'update', 'view'
	 * @return	int         					<0 if KO,
	 *                           				=0 if OK but we want to process standard actions too,
	 *                            				>0 if OK and we want to replace standard actions.
	 */
	public function getNomUrl($parameters, &$object, &$action)
	{
		global $db,$langs,$conf,$user;
		$this->resprints = '';
		return 0;
	}

	/**
	 * Overloading the doActions function : replacing the parent's function with the one below
	 *
	 * @param   array           $parameters     Hook metadatas (context, etc...)
	 * @param   CommonObject    $object         The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
	 * @param   string          $action         Current action (if set). Generally create or edit or null
	 * @param   HookManager     $hookmanager    Hook manager propagated to allow calling another hook
	 * @return  int                             < 0 on error, 0 on success, 1 to replace standard code
	 */
	public function doActions($parameters, &$object, &$action, $hookmanager)
	{
		global $conf, $langs, $user;

		$error = 0; // Error counter

		if (in_array($parameters['currentcontext'], array('propalcard', 'ordercard', 'invoicecard', 'ordersuppliercard', 'invoicesuppliercard', 'contractcard', 'projectcard', 'ticketcard', 'interventioncard', 'expeditioncard'))) {
			if (!empty($conf->global->BLOCKOUTSTANDING_ADDLINE) && ($action == 'addline' || $action == 'updateligne' || $action == 'updateline')) {
				$blockCards = array();
				if (! empty($conf->global->BLOCKOUTSTANDING_PROPOSAL)) {
					$blockCards[] = 'propalcard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_ORDER)) {
					$blockCards[] = 'ordercard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_SHIPMENT)) {
					$blockCards[] = 'expeditioncard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_INVOICE)) {
					$blockCards[] = 'invoicecard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_SUPPLIER_ORDER)) {
					$blockCards[] = 'ordersuppliercard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_SUPPLIER_INVOICE)) {
					$blockCards[] = 'invoicesuppliercard';
				}
				if (in_array($parameters['currentcontext'], $blockCards)) {
					$langs->load("blockoutstanding@blockoutstanding");
					$object->fetch_thirdparty();
					$soc = $object->thirdparty;
					if ($soc->id > 0) {
						$isSupplierObject = false;
						if (in_array($parameters['currentcontext'], array('ordersuppliercard','invoicesuppliercard'))) $isSupplierObject = true;
						$outstandingArray = $this->checkOutstandingLimit($object, $action, $soc, $isSupplierObject);
						if (!empty($outstandingArray)) {
							if ($conf->global->BLOCKOUTSTANDING_LEVEL == '1' && empty($user->rights->blockoutstanding->ignore)) {
								$this->errors[] = $langs->trans('BlockedOutstanding', $outstandingArray['outstandingLimit'], $outstandingArray['objectvalue'], $outstandingArray['outstandingbill']);
								$error++;
							} else {
								setEventMessage($langs->trans('BlockedOutstanding', $outstandingArray['outstandingLimit'], $outstandingArray['objectvalue'], $outstandingArray['outstandingbill']), 'warnings');
							}
						}
					}
				}
			}
			if (! empty($conf->global->BLOCKOUTSTANDING_CREATE) && ($action == 'create')) {
				$blockCards = array();
				if (! empty($conf->global->BLOCKOUTSTANDING_PROPOSAL)) {
					$blockCards[] = 'propalcard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_ORDER)) {
					$blockCards[] = 'ordercard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_SHIPMENT)) {
					$blockCards[] = 'expeditioncard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_INVOICE)) {
					$blockCards[] = 'invoicecard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_SUPPLIER_ORDER)) {
					$blockCards[] = 'ordersuppliercard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_SUPPLIER_INVOICE)) {
					$blockCards[] = 'invoicesuppliercard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_CONTRACT)) {
					$blockCards[] = 'contractcard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_PROJECT)) {
					$blockCards[] = 'projectcard';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_TICKET)) {
					$blockCards[] = 'ticketcard';
				}
				if (!empty($conf->global->BLOCKOUTSTANDING_INTERVENTION)) {
					$blockCards[] = 'interventioncard';
				}
				if (in_array($parameters['currentcontext'], $blockCards)) {
					$langs->load("blockoutstanding@blockoutstanding");
					$object->fetch_thirdparty();
					$soc = $object->thirdparty;
					if (empty($soc->id)) {
						$fk_soc = GETPOST('socid', 'int');
						if (empty($fk_soc)) $fk_soc = $parameters['socid'];
						if ($fk_soc > 0) {
							dol_include_once('societe/class/societe.class.php');
							$soc = new Societe($this->db);
							$soc->fetch($fk_soc);
						}
					}
					if (is_object($soc) && $soc->id > 0) {
						$isSupplierObject = false;
						if (in_array($parameters['currentcontext'], array('ordersuppliercard','invoicesuppliercard'))) $isSupplierObject = true;
						$outstandingArray = $this->checkOutstandingLimit($object, $action, $soc, $isSupplierObject);
						if (!empty($outstandingArray)) {
							if ($conf->global->BLOCKOUTSTANDING_LEVEL == '1' && empty($user->rights->blockoutstanding->ignore)) {
								$this->errors[] = $langs->trans('BlockedOutstanding', $outstandingArray['outstandingLimit'], $outstandingArray['objectvalue'], $outstandingArray['outstandingbill']);
								$error++;
							} else {
								setEventMessage($langs->trans('BlockedOutstanding', $outstandingArray['outstandingLimit'], $outstandingArray['objectvalue'], $outstandingArray['outstandingbill']), 'warnings');
							}
						}
					}
				}
			}
		}

		if (!$error) {
			return 0; // ok or warning
		} else {
			unset($_POST['price_ht']);
			unset($_POST['price_ttc']);
			unset($_POST['qty']);
			unset($_POST['lineid']);
			unset($_POST['idprodfournprice']);
			return -1;
		}
	}

	/**
	 * Overloading the formObjectOptions function : replacing the parent's function with the one below
	 *
	 * @param   array           $parameters     Hook metadatas (context, etc...)
	 * @param   CommonObject    $object         The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
	 * @param   string          $action         Current action (if set). Generally create or edit or null
	 * @param   HookManager     $hookmanager    Hook manager propagated to allow calling another hook
	 * @return  int                             < 0 on error, 0 on success, 1 to replace standard code
	 */
	public function formObjectOptions($parameters, &$object, &$action, $hookmanager)
	{
		global $conf, $user, $langs;

		$error = 0; // Error counter
		$errors = array();

		// block outstandlimit field
		if (in_array($parameters['currentcontext'], array('thirdpartycomm'))) {
			if (empty($user->rights->blockoutstanding->write)) {
				$langs->load("blockoutstanding@blockoutstanding");
				dol_include_once('/blockoutstanding/lib/blockoutstanding.lib.php');
				disableOutstandingLimitField($langs);
			}
		}
		// reload page for pages which do not reload on change company
		if (in_array($parameters['currentcontext'], array('contractcard', 'projectcard', 'ticketcard', 'ordersuppliercard', 'invoicesuppliercard'))) {
			if (! empty($conf->global->BLOCKOUTSTANDING_CREATE) && ($action == 'create' || $action == 'add')) {
				$blockCards = array();
				if (! empty($conf->global->BLOCKOUTSTANDING_CONTRACT)) {
					$blockCards[] = 'catchselectcompanyevent';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_PROJECT)) {
					$blockCards[] = 'catchselectcompanyevent';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_TICKET)) {
					$blockCards[] = 'catchselectcompanyevent';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_SUPPLIER_ORDER)) {
					$blockCards[] = 'catchselectcompanyevent';
				}
				if (! empty($conf->global->BLOCKOUTSTANDING_SUPPLIER_INVOICE)) {
					$blockCards[] = 'catchselectcompanyevent';
				}
				if (in_array('catchselectcompanyevent', $blockCards)) {
					print '<script>
					$(document).ready(function() {
						$("#socid").change(function() {
							var socid = $(this).val();
							var prjid = $("#projectid").val();
							// reload page
							window.location.href = "'.$_SERVER["PHP_SELF"].'?action=create&socid="+socid+"&projectid="+prjid
						});
					});
					</script>';
				}
			}
		}

		if (! $error) {
			return 0; // or return 1 to replace standard code
		} else {
			$this->errors[] = $errors;
			return -1;
		}
	}

	/**
	 * Overloading the moreHtmlStatus function : replacing the parent's function with the one below
	 *
	 * @param   array           $parameters     Hook metadatas (context, etc...)
	 * @param   CommonObject    $object         The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
	 * @param   string          $action         Current action (if set). Generally create or edit or null
	 * @param   HookManager     $hookmanager    Hook manager propagated to allow calling another hook
	 * @return  int                             < 0 on error, 0 on success, 1 to replace standard code
	 */
	public function moreHtmlStatus($parameters, &$object, &$action, $hookmanager)
	{
		global $conf, $user, $langs;

		$this->resprints = '';

		if (empty($conf->blockoutstanding->enabled)) return 0;

		if ($object && $object->element == 'societe' && $object->status == 1) {
			// check if blocked
			$outstandingArray = $this->checkOutstandingLimit(null, '', $object, false);
			if (!empty($outstandingArray)) {
				$langs->load("blockoutstanding@blockoutstanding");
				if ($conf->global->BLOCKOUTSTANDING_LEVEL == '1' && empty($user->rights->blockoutstanding->ignore)) {
					$statusType = 'status8';
				} else {
					$statusType = 'status2';
				}
				$this->resprints .= dolGetStatus($langs->transnoentitiesnoconv('BlockedThirdParty'), $langs->transnoentitiesnoconv('BlockedThirdParty'), '', $statusType, 2);
				if (!empty($conf->use_javascript_ajax) && !empty($user->rights->societe->creer) && !empty($conf->global->MAIN_DIRECT_STATUS_UPDATE)) {
					$this->resprints .= ' ' . ajax_object_onoff($object, 'status', 'status', 'BlockedThirdParty', 'ActivityCeased', array());
				}
				return 1;
			}
		}

		return 0;
	}

	/**
	 * Overloading the printFieldListOption function : replacing the parent's function with the one below
	 *
	 * @param   array           $parameters     Hook metadatas (context, etc...)
	 * @param   CommonObject    $object         The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
	 * @param   string          $action         Current action (if set). Generally create or edit or null
	 * @param   HookManager     $hookmanager    Hook manager propagated to allow calling another hook
	 * @return  int                             < 0 on error, 0 on success, 1 to replace standard code
	 */
	public function printFieldListOption($parameters, &$object, &$action, $hookmanager)
	{
		if (in_array($parameters['currentcontext'], array('thirdpartylist', 'customerlist'))) {
			$this->resprints = '<td class="liste_titre left">';
			$this->resprints .= '</td>';
		}
		return 0;
	}

	/**
	 * Execute action printFieldListTitle
	 *
	 * @param   array           $parameters     Array of parameters
	 * @param   CommonObject    $object         The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
	 * @param   string          $action         'add', 'update', 'view'
	 * @param   Hookmanager     $hookmanager    hookmanager
	 * @return  int                             <0 if KO,
	 *                                          =0 if OK but we want to process standard actions too,
	 *                                          >0 if OK and we want to replace standard actions.
	 */
	public function printFieldListTitle(&$parameters, &$object, &$action, $hookmanager)
	{
		global $langs;

		$this->resprints = '';

		if (in_array($parameters['currentcontext'], array('thirdpartylist', 'customerlist'))) {
			$langs->load("blockoutstanding@blockoutstanding");
			$this->resprints = '<th class="liste_titre left">'.$langs->trans('ObjectsToBlock').'</th>';
		}
		return 0;
	}

	/**
	 * Execute action printFieldListValue
	 *
	 * @param   array           $parameters     Array of parameters
	 * @param   CommonObject    $object         The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
	 * @param   string          $action         'add', 'update', 'view'
	 * @param   Hookmanager     $hookmanager    hookmanager
	 * @return  int                             <0 if KO,
	 *                                          =0 if OK but we want to process standard actions too,
	 *                                          >0 if OK and we want to replace standard actions.
	 */
	public function printFieldListValue(&$parameters, &$object, &$action, $hookmanager)
	{
		global $langs, $conf, $user;

		$this->resprints = '';

		if (in_array($parameters['currentcontext'], array('thirdpartylist', 'customerlist'))) {
			$this->resprints .= '<td>';
			$obj = $parameters['obj'];
			$soc = new Societe($this->db);
			$soc->fetch($obj->rowid);
			if ($soc->id > 0 && $soc->element == 'societe' && $soc->status == 1) {
				// check if blocked
				$outstandingArray = $this->checkOutstandingLimit(null, '', $soc, false);
				if (!empty($outstandingArray)) {
					$langs->load("blockoutstanding@blockoutstanding");
					if ($conf->global->BLOCKOUTSTANDING_LEVEL == '1' && empty($user->rights->blockoutstanding->ignore)) {
						$statusType = 'status8';
					} else {
						$statusType = 'status2';
					}
					$this->resprints .= dolGetStatus($langs->transnoentitiesnoconv('BlockedThirdParty'), $langs->transnoentitiesnoconv('BlockedThirdParty'), '', $statusType, 2);
				}
			}

			$this->resprints .= '</td>';
		}
		return 0;
	}

	/**
	 * checkOutstandingLimit : Check for outstanding limit for object and customer
	 *
	 * @param   CommonObject    $object           The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
	 * @param   string          $action           Current action (if set). Generally create or edit or null
	 * @param   Societe         $soc              Hook manager propagated to allow calling another hook
	 * @param   boolean         $isSupplierObject true is $object is supplier object
	 * @return  array                             array with check result, empty if no limit reached
	 */
	private function checkOutstandingLimit($object, $action, $soc, $isSupplierObject)
	{
		global $conf;

		$outstandingArray = array();
		$linePrice = 0;
		if ($object && !empty($object->total_ttc)) {
			$objectTotalTTC = $object->total_ttc;
		} else {
			$objectTotalTTC = 0;
		}

		if (!isset($soc->outstanding_limit)) {
			$outstandingLimit = price2num($conf->global->BLOCKOUTSTANDING_DEFAULT_AMOUNT, 'MT');
		} else {
			$outstandingLimit = $soc->outstanding_limit;
		}
		if (empty(GETPOST('tva_tx'))) {
			$tva_tx = 0;
		} else {
			$tva_tx = GETPOST('tva_tx');
		}
		if (!empty(GETPOST('price_ttc'))) {
			$priceTTC = GETPOST('price_ttc');
		} elseif (!empty(GETPOST('price_ht'))) {
			$priceTTC = GETPOST('price_ht') * (1 + $tva_tx/100);
		} else {
			$priceTTC = 0;
		}
		if (empty(GETPOST('qty'))) {
			$qty = 0;
		} else {
			$qty = GETPOST('qty');
		}
		if ($isSupplierObject) {
			if (empty($priceTTC)) {
				dol_include_once('fourn/class/fournisseur.product.class.php');
				$supplierProduct = new ProductFournisseur($this->db);
				if ($supplierProduct->fetch_product_fournisseur_price(GETPOST('idprodfournprice', 'int')) > 0) {
					$linePrice = $supplierProduct->fourn_price * $qty;
				}
			} else {
				$linePrice = $priceTTC * $qty;
			}
			$mode = 'supplier';
		} else {
			$mode = 'customer';
			$linePrice = $priceTTC * $qty;
		}
		$arrayoutstandingbills = $soc->getOutstandingBills($mode);
		$outstandingbill = $arrayoutstandingbills['opened'];
		if (($action == 'updateligne' || $action == 'updateline') && is_array($object->lines)) {
			$lineId = GETPOST('lineid');
			foreach ($object->lines as $line) {
				if ($line->id == $lineId) {
					$linePrice -= $line->total_ttc;
				}
			}
		}
		if ($outstandingbill > 0 &&
			(isset($soc->outstanding_limit) || !empty($outstandingLimit)) &&
			$outstandingbill + $objectTotalTTC + $linePrice > $outstandingLimit) {
				$outstandingArray['outstandingLimit'] = price($outstandingLimit);
				$outstandingArray['objectvalue'] = price($objectTotalTTC + $linePrice);
				$outstandingArray['outstandingbill'] = price($outstandingbill);
		}
		return $outstandingArray;
	}
}
